/**
* \file: device_metadata.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>

#include "model/device_metadata.h"

typedef struct device_metadata_t
{
	char *device_type;
	char *identifier;
	device_metadata_free_func_t free_func;
} device_metadata_t;

static void device_metadata_default_free(device_metadata_t *metadata);

static void device_metadata_free_elements(device_metadata_t *metadata);

//-------------------------------------------- API members -------------------------------------------------------
device_metadata_t *device_metadata_new(const char *device_type, const char *identifier)
{
	return device_metadata_flexible_new(device_type,identifier,
			sizeof(device_metadata_t), device_metadata_default_free);
}

device_metadata_t *device_metadata_flexible_new(const char *device_type, const char *identifier,
		size_t metadata_size, device_metadata_free_func_t free_func)
{
	device_metadata_t *metadata;
	metadata=malloc(metadata_size);
	if (metadata!=NULL)
	{
		metadata->device_type=strdup(device_type);
		metadata->identifier=strdup(identifier);
		metadata->free_func=free_func;

		if (metadata->device_type==NULL || metadata->identifier==NULL)
		{
			device_metadata_free_elements(metadata);
			free(metadata);
			metadata=NULL;
		}
	}

	return metadata;
}

void device_metadata_free(device_t *device)
{
	device_metadata_t *metadata;
	device_metadata_free_func_t free_func;

	metadata=device_get_metadata(device);
	free_func=metadata->free_func;

	device_metadata_free_elements(metadata);

	free_func(metadata);
}

const char *device_metadata_get_device_type(device_t *device)
{
	device_metadata_t *metadata;
	metadata=device_get_metadata(device);
	return metadata->device_type;
}

const char *device_metadata_get_device_identifier(device_t *device)
{
	device_metadata_t *metadata;
	metadata=device_get_metadata(device);
	return metadata->identifier;
}

//----------------------------------------------------------------------------------------------------------------

//-------------------------------------- private members ---------------------------------------------------------
static void device_metadata_free_elements(device_metadata_t *metadata)
{
	//sorry, this is needed to get rid of lint finding: "Custodial pointer ctx (line xxx) has not been freed or returned"
	device_metadata_t *metadata_cp=metadata;

	if (metadata_cp->device_type!=NULL)
		free(metadata_cp->device_type);
	if (metadata_cp->identifier!=NULL)
		free(metadata_cp->identifier);
}

static void device_metadata_default_free(device_metadata_t *metadata)
{
	free(metadata);
}
//----------------------------------------------------------------------------------------------------------------
